/*
 * Table-generator
 *
 * Sourcecode for generating lookup-tables to "Raycast demo #3". Should be
 * cross-platform, wrote it in Linux if anybody wonders.
 */

#include <stdio.h>
#include <math.h>

/* Some configurations */
#define CIRCLE		256
#define QUARTER		(CIRCLE/4)
#define HALF		(CIRCLE/2)
#define QUARTER3	(HALF + QUARTER)
#define FOV			64
#define BYTE_SIZE	256
#define WALL_HEIGHT	360

#define ABS(x)	(x < 0 ? -x : x)

/* Most tables */
int	tantbl[CIRCLE + QUARTER],
	dtbl[CIRCLE + QUARTER],
	otbl[CIRCLE + QUARTER],
	fhtbl[BYTE_SIZE / 2],
	shtbl[BYTE_SIZE],
	movtbl[40];

FILE	*file;

int main() {
	/* Hmm, why aren't those global too? */
	int	i, j, k;
	float	tang;
	float	s,
			new,
			old;

	/* Tracing-tables */

	for (i = 0; i < CIRCLE + QUARTER; i++) {
		tang = tan(i * M_PI / HALF);

		tantbl[i] = (int)(256 * tang);
		if (i > QUARTER && i < QUARTER3) tantbl[i] = -tantbl[i];
		if (i == QUARTER || i == QUARTER3) tantbl[i] = 0;

		dtbl[i] = (int)(256 * sqrt(tang * tang + 1));
		if (i == QUARTER || i == QUARTER3) dtbl[i] = 32767;

		otbl[i] = (int)ABS(255 * sin(i * M_PI / HALF));
	}

	/* fhtbl -> fine-heights table   shtbl -> simple-heights table */

	for (i = 0; i < BYTE_SIZE / 2; i++) {
		fhtbl[i] = (int)(WALL_HEIGHT / (0.25f*i + 1));
		if (fhtbl[i] > 60) fhtbl[i] = 60;
		if (!fhtbl[i]) fhtbl[i] = 1;
	}
	for (i = 0; i < BYTE_SIZE; i++) {
		shtbl[i] = (int)(WALL_HEIGHT / (3*i + 33));
		if (shtbl[i] > 60) shtbl[i] = 60;
		if (!shtbl[i]) shtbl[i] = 1;
	}

	/* For movement */

	for (i = 0; i < 40; i++) movtbl[i] = (int)(32 * sin(i * M_PI / 16));

	if (!(file = fopen("tbls.inc", "w"))) return 0;

	fprintf(file, "tantbl:");
	j = 0;
	for (i = 0; i < CIRCLE + QUARTER; i++) {
		if (!j) fprintf(file, "\n\t.dw\t");
		fprintf(file, "%d", tantbl[i]);
		if (j < 15) fprintf(file, ", ");
		j = (j + 1) & 15;
	}

	fprintf(file, "\n\ndtbl:");
	j = 0;
	for (i = 0; i < CIRCLE + QUARTER; i++) {
		if (!j) fprintf(file, "\n\t.dw\t");
		fprintf(file, "%d", dtbl[i]);
		if (j < 15) fprintf(file, ", ");
		j = (j + 1) & 15;
	}

	fprintf(file, "\n\notbl:");
	j = 0;
	for (i = 0; i < CIRCLE + QUARTER; i += 2) {
		if (!j) fprintf(file, "\n\t.db\t");
		fprintf(file, "%d", otbl[i]);
		if (j < 15) fprintf(file, ", ");
		j = (j + 1) & 15;
	}

	fprintf(file, "\n\nheights:");
	j = 0;
	for (i = 0; i < BYTE_SIZE / 2; i++) {
		if (!j) fprintf(file, "\n\t.db\t");
		fprintf(file, "%d", fhtbl[i]);
		if (j < 15) fprintf(file, ", ");
		j = (j + 1) & 15;
	}
	j = 0;
	for (i = 0; i < BYTE_SIZE / 2; i++) {
		if (!j) fprintf(file, "\n\t.db\t");
		fprintf(file, "%d", shtbl[i]);
		if (j < 15) fprintf(file, ", ");
		j = (j + 1) & 15;
	}

	/* Wrote this directly to the file
	 * A texture-optimization from the last demo, which is the main reason
	 * for the size of this demo. Instead of doing v += vs all the time, I
	 * store all possible changes on v in this new table.
	 */

	/* Pointers to v-change table */
	fprintf(file, "\n\nvptbl:");
	j = 0;
	for (i = 1; i <= 60; i++) {
		if (!j) fprintf(file, "\n\t.dw\t");
		fprintf(file, "VS%d", i);
		if (j < 15) fprintf(file, ", ");
		j = (j + 1) & 15;
	}

	/* V-changes */
	fprintf(file, "\n\nvstbl:");
	for (i = 1; i <= 60; i++) {
		fprintf(file, "\nVS%d\t.db\t", i);
		s = 32.0f / (2 * (i - 1));
		if (s > 32) s = 32;
		old = 0;
		new = i > 25 ? s * (i - 24) : s;
		for (j = 0; j < (i > 24 ? 48 : 2 * i) - 1; j++) {
			fprintf(file, "%d", 4 * ((int)new - (int)old));
			old = new;
			new += s;
			if (j == 30) fprintf(file, "\t\\\t.db\t");
			else {
				if (i > 24) {
					if (j < 46) fprintf(file, ", ");
				}
				else if (j < 2 * i - 2) fprintf(file, ", ");
			}
		}
	}

	fprintf(file, "\n\nmovtbl:");
	j = 0;
	for (i = 0; i < 40; i++) {
		if (!j) fprintf(file, "\n\t.db\t");
		fprintf(file, "%d", movtbl[i]);
		if (j < 15) fprintf(file, ", ");
		j = (j + 1) & 15;
	}

	/* Useful bitmasks */
	fprintf(file, "\n\nmasks:\n");
	fprintf(file, "\t.db\t128, 64, 32, 16, 8, 4, 2, 1");

	fclose(file);

	return 0;	
}
