;##################################################################
;
;   Phoenix III - Collisions between enemies and player's bullets
;
;   Programmed by Patrick Davidson (pad@calc.org)
;        
;   This program is in the public domain.  There is no warranty.
;
;   This file was last modified May 18, 2001.  
;
;##################################################################

;############## Builds table of enemies in TEMP_AREA
;
; This puts enemies into columns, 0 to 7, on the high bits of their X
; coordinates.  Column 0 is for X coordinates 0 to 15, 1 for 16 to 31, and
; so on.  Then, they are split into groups; for enemies in only 1 column,
; the group is column * 2 + 1, for enemies that spill into the next column,
; the group is column * 2 + 2.
;
; Each group data structure has a one-byte size, followed by a list of
; pointers to each enemy in the group.  The first column starts at
; TEMP_AREA+64, the next at TEMP_AREA+128, and so on.  Empty columns are put
; at the edges to allow proper checking of side bullets.  This allows up to
; 31 entries per column, more than enough to accomodate all enemies.

hit_enemies:
        xor     a                       ; Initialize all columns as empty
        ld      b,19
        ld      de,64
smc_test_1:
        ld      hl,TEMP_AREA
loop_empty_columns:
        ld      (hl),a
        add     hl,de
        djnz    loop_empty_columns

        ld      hl,enemies              ; HL -> enemy
        ld      b,e_num                 
build_table_loop:
        ld      a,(hl)                  ; Skip empty enemy entries
        add     a,a
        jr      z,none_here

        inc     hl
        ld      a,(hl)                  ; A = enemy X coordinate
        inc     hl
        ld      e,(hl)                  ; E = enemy width
        dec     hl
        dec     hl                      ; HL -> start of enemy

        push    hl
        call    which_group             ; Compute group number in A (0-18)

        ld      a,(hl)                  ; A = group size before insertion
        inc     (hl)                    ; increment group size
        add     a,a
        inc     a            
        call    ADD_HL_A                ; HL = new entry in group     

        pop     de                      ; DE -> enemy
        ld      (hl),e                  ; store DE in group
        inc     hl
        ld      (hl),d
        ex      de,hl                   ; HL -> enemy

none_here:
        ld      de,e_size
        add     hl,de
        djnz    build_table_loop

;############## Process player bullets
;
; This routine will process every player bullet by checking it for collisions
; with appropriate columns.  If it is all in one column, it must be tested
; against three groups, as shown below
;
; Column 2     |     Column 3      |     Column 4
;              |                   |
;               <-----Bullet------>
;     <---Enemy group 6--->
;               <--Enemy group 7-->
;                         <---Enemy group 8--->
;
; If the bullet straddles two columns, it then must be checked against
; five groups of enemies, as indicated below
;
; Column 2     |    Column 3       |     Column 4     |      Column 5
;              |                   |                  |
;                      <--------Bullet-------->
;    <---Enemy group 6--->
;               <--Enemy group 7-->
;                      <-----Enemy group 8---->
;                                   <-Enemy group 9-->
;                                           <---Enemy group 10--->
;
; Even so, this still saves a lot of time over the default collision testing
; strategy, to test all combinations of objects, which would effectively
; require testing with 15 groups in all cases (though without the sorting
; overhead introduced here).

        ld      hl,bullets
        ld      b,b_count

loop_process_bullets:
        ld      a,(hl)
        or      a          
        jr      z,no_bullet

        push    hl
        push    bc

        ld      (collision+1),hl
        inc     hl
        ld      a,(hl)
        inc     hl
        ld      e,(hl)
        dec     hl

        push    hl
        call    which_group             ; HL -> group data
        ex      (sp),hl

        ld      de,test_coords          ; Copy bullet data to test coords
        ldi
        ldi
        ldi
        ldi

        pop     hl

        bit     0,b
        jr      nz,check_3_groups

check_5_groups:
        call    list_collision_check
        ld      bc,-64
        add     hl,bc
        call    list_collision_check
        ld      bc,128
        add     hl,bc
check_3_groups:
        call    list_collision_check
        ld      bc,64
        add     hl,bc
        call    list_collision_check
        ld      bc,64
        add     hl,bc
        call    list_collision_check

bullet_done:
        pop     bc
        pop     hl

no_bullet:
        ld      de,b_size
        add     hl,de
        djnz    loop_process_bullets
        ret

;############## Determine which group object is in
;
; Determines which group an object is in.  A holds the X coordinate, E holds
; the width.  Returns group pointer in HL.  Modifies A, C, D, and HL.

which_group:
        ld      d,a
        rlca
        rlca
        rlca
        rlca
        and     7
        add     a,a
        inc     a
        ld      c,a
        ld      a,d
        and     15
        add     a,e
        cp      16
        ld      a,c
        jr      c,no_wg_inc
        inc     a
no_wg_inc:

#ifdef __TI83__
        add     a,a
        add     a,a
        add     a,a
        ld      l,a
        ld      h,0
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      de,TEMP_AREA
        add     hl,de
#else
        rrca                            ; RRCAs compute group * 64
        rrca                            ; bits 6-7 are top of low byte
        ld      h,a                     ; bits 3-5 are 0
        and     %11000000               ; bits 0-2 are bottom of high byte 
        ld      l,a
        ld      a,h                     ; or will add $e0 to A (since this
        or      $e0                     ; always sets the top two bits, the
        ld      h,a                     ; previous value doesn't matter)
#endif
        ret

;############## Check bullet for collisions within one group
;
; Checks the bullet whose data is int test_coords with group (HL)

list_collision_check:
        push    hl
        ld      a,(hl)
        or      a
        jr      z,exit_cc_list
        inc     hl                  ; HL -> group data
        ld      b,a                 ; B = number in group

loop_column_scan:
        push    hl                  ; HL -> somewhere in the list
        ld      a,(hl)
        inc     hl
        ld      h,(hl)
        ld      l,a                 ; HL -> item retrieved from list
        ld      de,e_x
        push    hl
        add     hl,de
        call    collision_check
        pop     hl
        jr      c,collision
        pop     hl
        inc     hl                  ; Move HL to next in list
        inc     hl
        djnz    loop_column_scan

exit_cc_list:
        pop     hl
        ret

collision:
        ld      de,0                ; DE -> bullet
        ex      de,hl               ; DE -> enemy, HL -> bullet 

        ld      (hl),0              ; Delete bullet
        ld      bc,b_misc
        add     hl,bc
        ld      c,(hl)              ; C = bullet damage

        call    damage_enemy

        pop     hl
        pop     hl
        pop     hl
        jr      bullet_done

;############## Do C points of damage to enemy at (DE)

damage_enemy:
        ld      hl,e_dmg
        add     hl,de               ; HL -> enemy damage
        ld      a,(hl)
        sub     c
        ld      (hl),a
        ret     nc                  ; If enemy not destroyed

        ex      de,hl

        ld      (hl),128
        ld      bc,e_path_pos
        add     hl,bc
        ld      (hl),(explosion_data & 255)
        inc     hl
        ld      (hl),((explosion_data>>8) & 255)
        inc     hl
        ld      (hl),30
        inc     hl
        ld      (hl),(explosion_img & 255)
        inc     hl
        ld      (hl),((explosion_img>>8) & 255)
        inc     hl
        ld      (hl),5
        inc     hl
        inc     hl
        inc     hl
        ld      (hl),0
        ld      a,$75
        ld      de,number+2
        ld      (de),a
        ld      hl,score+2
        jp      ADD_BCD

;############## Explosion image data

explosion_data:
        .db     0,0
        .db     0

explosion_img:
        .dw     explode1
        .db     5
        .dw     explode2
        .db     5
        .dw     explode3
        .db     127
        .dw     explode4

explode1:
        .db     6
        .db     %00000000       
        .db     %00011100       
        .DB     %00111110       
        .DB     %01010110       
        .DB     %00111000       
        .DB     %00000000

        .db     %00110000       
        .db     %01001110       
        .DB     %10111110       
        .DB     %01001111       
        .DB     %00111000       
        .DB     %00011010       

explode2:
        .db     6
        .db     %10001011       
        .db     %01001110       
        .DB     %10000001       
        .DB     %01000101       
        .DB     %00101110       
        .DB     %11111010       
                        
        .db     %11110011       
        .db     %01001110       
        .DB     %10000001       
        .DB     %01000101       
        .DB     %00110110       
        .DB     %11011010

explode3:
        .db     6     
        .db     %01000001       
        .db     %00100110       
        .DB     %00010101       
        .DB     %01000100       
        .DB     %00010010       
        .DB     %10011010       
        
        .db     %01000010       
        .db     %00100000       
        .DB     %00000001       
        .DB     %01000100       
        .DB     %00100010       
        .DB     %10001010

explode4:
        .db     6     
        .db     %00001000       
        .db     %11000010       
        .DB     %00000000       
        .db     %00100000       
        .db     %00000001       
        .db     %00110000       
         
        .db     %00000100       
        .DB     %00000000       
        .DB     %01000000       
        .DB     %00000000       
        .db     %00000001       
        .db     %00100100
