;##################################################################
;
;   Phoenix III for the TI-83+ - low level display/interrupt/background/keys
;
;   Programmed by Patrick Davidson (pad@calc.org)
;        
;   This program is in the public domain.  There is no warranty.
;
;   This file was last updated May 27, 2001.
;
;##################################################################

;############## Interrupt routine
;
; Advances grayscale frame to the next one to be displayed.  Note that the
; next frame begins display one frame after the video address port is
; updated.
;
; After a frame has been show a sufficient number of times, the interrupt
; automatically switches to the other frame, and the rest of the program
; must keep up.  This is required due to the one-frame delay, since it is
; quite difficult to predict whether the main program will finish drawing
; one frame later.
;
; Since the cycle counter is reset to 0 after the display port is set, the
; new buffer address will only be displayed on the next interrupt, after
; which the cycle is set to 1.  Due to the one-frame delay, the next buffer
; only starts being shown when the cycle is 2.

timer_interrupt:
        push    hl
        push    af
        ld      hl,num_cycles
        inc     (hl)
        pop     af
        pop     hl
        JP      $38
timer_interrupt_end:

;############## Initialization

init:   ld      hl,timer_interrupt
        ld      de,interrupt_entry
        ld      bc,timer_interrupt_end-timer_interrupt
        ldir

        ld      hl,interrupt_table
        ld      (hl),interrupt_byte
        ld      bc,256
        ld      d,h
        ld      e,l
        inc     de
        ldir

        ld      a,interrupt_reg
        ld      i,a

        im      2

        ei

        ld      hl,TEXT_MEM            ; zero variables
        ld      de,TEXT_MEM+1
        xor     a
        ld      (hl),a
        ld      bc,127
        ldir

        ld      (exit_program+1),sp
        call    phoenix3
exit_program:
        ld      sp,0
        ld      a,(game_over)
        or      a
        ret     z
        jp      do_scoring

;############## Show a grayscale frame

twocolor_white:
        call    wait_change
        ld      a,(hl)
        cp      5
        jr      c,twocolor_white
        ld      hl,main_plane0
        ld      (hl),0
        jr      copy_screen

show_frame:
        ld      hl,num_cycles
        jr      grayscale
        jr      twocolor_black
        jr      twocolor_white

grayscale:
        call    wait_frame_end
        ld      hl,main_plane1
        call    copy_screen
        call    wait_change
        ld      hl,main_plane0


;############## Screen buffer copy routine

copy_screen:
        ld      a,$80           ; 7
        out     ($10),a         ; 11
        ld      de,(16*64)-16-1
        add     hl,de
        ld      a,$20           ; 7
        ld      c,a             ; 4     ; 43
fastCopyAgain:
        ld      b,64            ; 7
        inc     c               ; 4
        ld      de,-(16*64)+1   ; 10
        nop
        nop
        out     ($10),a         ; 11
        add     hl,de           ; 11
        ld      de,15           ; 10
fastCopyLoop:
        nop
        nop
        add     hl,de           ; 11
        inc     hl              ; 6
        ret     c               ; 5
        ld      a,(hl)          ; 7
        out     ($11),a         ; 11
        djnz    fastCopyLoop    ; 13/8  ; 3392
        ld      a,c             ; 4
        cp      $2B+1           ; 7
        jr      nz,fastCopyAgain; 12/7
        ret     ; 11    ; 18

;############## Show frame (black & white, light gray -> black)

twocolor_black:
        call    wait_change
        ld      a,(num_cycles)
        cp      5
        jr      c,twocolor_black
        ld      hl,main_plane0
        ld      (hl),0

;############## Screen buffer copy routine (ORs two planes together)

        ld      a,$80           ; 7
        out     ($10),a         ; 11
        ld      de,(16*64)-16-1
        add     hl,de
        ld      a,$20           ; 7
        ld      c,a             ; 4     ; 43
orCopyAgain:
        ld      b,64            ; 7
        inc     c               ; 4
        ld      de,-(16*64)+1   ; 10
        nop
        nop
        out     ($10),a         ; 11
        add     hl,de           ; 11
        ld      de,16           ; 10
orCopyLoop:

        add     hl,de
        ld      a,(hl)
        inc     h
        inc     h
        inc     h
        inc     h
        or      (hl)
        dec     h
        dec     h
        dec     h
        dec     h

        out     ($11),a         ; 11
        djnz    orCopyLoop    ; 13/8  
        ld      a,c             ; 4
        cp      $2B+1           ; 7
        jr      nz,orCopyAgain; 12/7
        ret     ; 11    ; 18

;############## Wait for next frame, and do page swap

wait_frame_end:
        call    wait_change
        ld      a,(hl)
        cp      4
        jr      c,wait_frame_end
        ld      (hl),0
        ret

wait_change:
        ld      hl,num_cycles
        ld      a,(hl)
loop_wait:
        cp      (hl)
        jr      z,loop_wait
        ret

;############## Display a frame from the scrolled screen

;----E800
;
; size here in noscroll_size (this is *bottom* part of screen)
;
;----scrolling line
;
; size here in scroll_size (this is *top* part of screen)
;
;----EBFF

display_scrolled:
        ld      a,(scroll_line)
        or      a
        jr      z,scroll_offset_0

        add     a,a
        add     a,a
        ld      l,a
        ld      h,0
        add     hl,hl
        add     hl,hl                   ; HL = scroll amount * 16
        ld      (scroll_size),hl

        ex      de,hl
        ld      hl,$400
        sbc     hl,de
        ld      (noscroll_size),hl

        ld      de,background_plane0+3  ; copy top of plane 0
        add     hl,de
        ld      de,main_plane0
        ld      a,(scroll_line)
        call    copy_a_lines

        ld      bc,-$400                ; copy bottom of plane 0
        add     hl,bc
        ld      a,(scroll_line)
        neg
        and     63
        call    copy_a_lines

        ld      hl,(noscroll_size)      ; copy top of plane 1
        ld      de,background_plane1+3
        add     hl,de
        ld      de,main_plane1
        ld      a,(scroll_line)
        call    copy_a_lines

        ld      bc,-$400                ; copy bottom of plane 1
        add     hl,bc
        ld      a,(scroll_line)
        neg
        and     63

copy_a_lines:
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        ldi
        inc     hl
        inc     hl
        inc     hl
        inc     hl
        inc     de
        inc     de
        inc     de
        inc     de
        dec     a
        jr      nz,copy_a_lines
        ret

scroll_offset_0:
        ld      hl,background_plane0+3
        ld      de,main_plane0
        ld      a,64
        jr      copy_a_lines

;############## Scroll the screen up one line

scroll_screen:
        ld      a,(game_timer)
        rra
        ret     c
do_scroll_screen:
        ld      a,(scroll_line)
        inc     a
        and     63
        ld      (scroll_line),a

        ld      hl,ycoord
        dec     (hl)
        ld      a,(hl)
        cp      -1
        jr      nz,no_reset_map_pos

        ld      hl,map_end
        ld      de,-13
        add     hl,de
        ld      (map_pos),hl

no_reset_map_pos:
        and     7
        ld      c,a
        ld      de,(map_pos)
        ld      a,(scroll_line)
        call    render_background_line

        ld      a,(ycoord)
        and     7
        ret     nz
        ld      hl,(map_pos)
        ld      de,-13
        add     hl,de
        ld      (map_pos),hl
        ret

;############## Render background at (DE) into line A (segment c)

render_background_line:
        dec     a
        cpl
        add     a,a
        add     a,a
        ld      l,a                  
        inc     l                    
        ld      h,0
        add     hl,hl
        add     hl,hl

        push    de
        ld      de,background_plane0
        add     hl,de
        pop     de
        dec     hl

        ld      b,12
loop_render_background:
        ld      a,(de)                  ; A = tile #
        inc     de
        add     a,a
        add     a,a
        add     a,a
        add     a,a                     ; A = offset of this tiles
        add     a,c                     ; A = offset of this line in tile

        push    de
        push    hl
        ld      e,a
        ld      d,0
        ld      hl,tiles
        add     hl,de                   ; HL -> tile data
        ld      a,(hl)                  ; A = tile data (plane 0)
        ld      de,8
        add     hl,de
        ld      d,(hl)                  ; H = tile data (plane 1)
        pop     hl
        ld      (hl),a                  ; Write plane 0 data
        inc     h
        inc     h
        inc     h
        inc     h
        ld      (hl),d
        dec     h
        dec     h
        dec     h
        dec     h
        pop     de
        inc     hl
        djnz    loop_render_background
        ret

;############## Scroll in the top of the backgroudn

initialize_background:
        ld      hl,scroll_line
        ld      a,-64
        add     a,(hl)
        ld      (hl),a

        ld      b,64
pscr:   push    bc
        call    do_scroll_screen
        pop     bc
        djnz    pscr
        ret

;############## GET_KEY replacement

SUPER_GET_KEY:
        push    hl
        push    de
        push    bc
        ld      e,0                     ; E = GET_KEY result
        ld      hl,getkeylastdata       ; HL = ptr to last read's table
        ld      a,$fe                   ; A = key port mask
        ld      d,1                     ; D = individual key's mask
        ld      c,0                     ; C = key number counter
gkol:   out     (1),a
        ld      b,8                         
        push    af
gkl:    inc     c
        in      a,(1)
        and     d
        jr      nz,nokey
        ld      a,(hl)
        and     d
        jr      z,nokey
        ld      e,c
nokey:  rlc     d
        djnz    gkl
        in      a,(1)
        ld      (hl),a
        pop     af
        inc     hl
        rlca     
        cp      $7F
        jr      nz,gkol
        ld      a,e
        pop     bc
        pop     de
        pop     hl
        ret

getkeylastdata:
        .db     $ff,$ff,$ff,$ff,$ff,$ff,$ff

;############## Low-level support routines

ADD_HL_A:
        add     a,l
        ld      l,a
        ret     nc
        inc     h
        ret

_ldHLind:
LD_HL_MHL:
        ld      a,(hl)
        inc     hl
        ld      h,(hl)
        ld      l,a
        ret

CP_HL_DE:
        push    hl
        and     a
        sbc     hl,de
        pop     hl
        ret
